<?php
/**
* LICENCIA
*
* Este programa se propociona "tal cual", sin garantía de ningún tipo más allá del soporte
* pactado a la hora de adquirir el programa.
*
* En ningún caso los autores o titulares del copyright serán responsables de ninguna
* reclamación, daños u otras responsabilidades, ya sea en un litigio, agravio o de otro
* modo, que surja de o en conexión con el programa o el uso u otro tipo de acciones
* realizadas con el programa.
*
* Este programa no puede modificarse ni distribuirse sin el consentimiento expreso del autor.
*
*    @author    Carlos Fillol Sendra <festeweb@festeweb.com>
*    @copyright 2014 Fes-te web! - www.festeweb.com
*    @license   http://www.festeweb.com/static/licenses/fs2ps_1.1.0.txt
*/

include_once(dirname(__FILE__).'/Fs2psDto2RowMatcher.php');
include_once(dirname(__FILE__).'/Fs2psException.php');
include_once(dirname(__FILE__).'/Fs2psTools.php');


class Fs2psCategoryMatcher extends Fs2psDto2TermMatcher
{
    public function __construct($task, $entity) {
        parent::__construct('categories', 'product_cat');
        $this->reloadCfg($task->cfg);
    }
}

class Fs2psSectionExtractorMatcher extends Fs2psCategoryMatcher
{
    public function _dtoIdStrFromRowId($row_id)
    {
        // Si no hay match, generamos un dto_id numérico a partir del row_id con una F delante
        $dto_id = parent::_dtoIdStrFromRowId($row_id);
        return empty($dto_id)? sprintf('S%03d', $this->generateDtoIdStrFromRowId($row_id % 1000)) : $dto_id;
    }
}

class Fs2psFamilyExtractorMatcher extends Fs2psCategoryMatcher
{
    public function _dtoIdStrFromRowId($row_id)
    {
        // Si no hay match, generamos un dto_id numérico a partir del row_id con una F delante
        $dto_id = parent::_dtoIdStrFromRowId($row_id);
        return empty($dto_id)? sprintf('F%03d', $this->generateDtoIdStrFromRowId($row_id % 1000)) : $dto_id;
    }
}

class Fs2psTaxonomyMatcher extends Fs2psDto2TermMatcher
{
    public function __construct($task, $entity, $taxonomy) {
        parent::__construct($entity, $taxonomy);
        $this->reloadCfg($task->cfg);
    }
    
    public function _dtoIdStrFromRowId($row_id)
    {
        // TODO: Revisar
        // Si no hay match, generamos un dto_id numérico a partir del row_id
        $dto_id = parent::_dtoIdStrFromRowId($row_id);
        return empty($dto_id)? $this->generateDtoIdStrFromRowId($row_id) : $dto_id;
    }
}

class Fs2psManufacturerMatcher extends Fs2psTaxonomyMatcher
{
    public function __construct($task, $entity, $taxonomy='pwb-brand') {
        // pwb-brand es la taxonomía que usa "Perfect WooCommerce Brands"
        // Se puede sobreescribir la taxonomía a usar a través de la configuración. Por ejemplo:
        // MANUFACTURERS_TAXONOMY = pa_marcas
        parent::__construct($task, 'manufacturers', $taxonomy);
    }
}

class Fs2psSupplierMatcher extends Fs2psTaxonomyMatcher
{
    public function __construct($task, $entity, $taxonomy=null) {
        // Se puede indicar la taxonomía a usar a través de la configuración
        parent::__construct($task, $entity, $taxonomy);
    }
}

class Fs2psGroupMatcher extends Fs2psDto2RowDirectMatcher
{
    public function __construct($task, $entity) {
        // table = null. Los roles no se almacenan en una tabla de bd al uso
        parent::__construct($entity, null, 'role', array('ref'), true, true);
        $this->reloadCfg($task->cfg);
    }
    
    /**
     * Se generan errores del tipo 'has_cap was called with an argument' si dtoId=rowId es una string formada por dígitos numéricos.
     * Lo evitamos poniendo el prefijo 'fw-' en el nombre del rol cuando esto sucede.
     * Simplificamos porque fields va a ser siempre array('ref').
     */
    public function _rowIdFromOid($oid) 
    {
        if (empty($oid)) return null;
        $row_id = strtolower(strval($oid));
        if (ctype_digit($row_id)) $row_id = sprintf('fw-%s', $row_id);
        return $row_id;
    }
    public function _oidStrFromRowId($row_id) 
    { 
        if (empty($row_id)) return null;
        if (substr($row_id, 0, 3)==='fw-') return substr($row_id, 3);
        return $row_id; 
    }
    
    public function existsRowInDatabase($id)
    {
        return !empty($id) && get_role($id);
    }

}

class Fs2psAttributeGroupMatcher extends Fs2psDto2RowMatcher
{
    protected $several_row_ids_allowed = false;
    
    public function __construct($task, $entity, $several_row_ids_allowed = false)
    {
        $this->several_row_ids_allowed = $several_row_ids_allowed;
        parent::__construct($entity, 'woocommerce_attribute_taxonomies', 'attribute_id', array('ref'), !$several_row_ids_allowed);
        $this->reloadCfg($task->cfg);
    }
    
    public function rowIdFromDtoId($dto_id)
    {
        $row_id = parent::rowIdFromDtoId($dto_id);
        if ($row_id && is_string($row_id)) {
            $row_id = array_map('intval', preg_split("/ *, */", $row_id));
            if (sizeof($row_id)==1) $row_id = $row_id[0];
            else if (sizeof($row_id)>1 && !$this->several_row_ids_allowed) {
                throw new Fs2psException(
                    'No se permite '.$dto_id.' mapeado con varios grupos: '.implode(',', $row_id)
                );
            }
        }
        return $row_id;
    }
    
    public function attributeNameFromDtoId($dto_id) {
        $dto_id_str = $this->dtoIdToStr($dto_id);
        return $dto_id_str=='SIZES'? 'size': ($dto_id_str=='COLOURS'? 'color' : preg_replace('/s$/', '', strtolower($dto_id_str)));
    }
    
    public function taxonomyFromDtoId($dto_id) {
        $attribute_group_id = $this->rowIdFromDtoId($dto_id);
        if ($attribute_group_id && !is_array($attribute_group_id)) { // Si es un array será porque estamos exportando de varios attribute groups
            $taxonomy = Fs2psTools::dbValue("
    			SELECT concat('pa_', attribute_name)
                FROM `@DB_woocommerce_attribute_taxonomies`
                where attribute_id=".$attribute_group_id."
    		");
        } else {
            $taxonomy = 'pa_'.(strtolower($dto_id)=='sizes'? 'size' : 'color');
        }
        return $taxonomy;
        
    }
    
    public function _rowIdFromDtoId($dto_id) {
        $row_id = Fs2psTools::dbValue('
            SELECT attribute_id
            FROM `@DB_woocommerce_attribute_taxonomies`
            where attribute_name= \''.$this->attributeNameFromDtoId($dto_id).'\'
        ');
         return $row_id!==null? intval($row_id) : null;
    }

}

class Fs2psAttributeMatcher extends Fs2psDto2TermMatcher
{
    protected $task;
    
    public function __construct($task, $name) { // $name = 'sizes' or 'colours'
        $this->task = $task;
        $attribute_groups_matcher = Fs2psMatcherFactory::get($task, 'attribute_groups');
        parent::__construct($name, $attribute_groups_matcher->taxonomyFromDtoId(strtoupper($name)));
        $this->reloadCfg($task->cfg);
    }
}

class Fs2psAttributeExtractorMatcher extends Fs2psAttributeMatcher
{
    protected $dto_id_from_name_replacex;
    protected $dto_id_from_name_patterns;
    
    public function reloadCfg($cfg) {
        parent::reloadCfg($cfg);
        $this->dto_id_from_name_replacex = $cfg->get(
            'IMATCH_'.strtoupper($this->entity).'_NAME_REPLACEX',
            array()
        );
        $this->dto_id_from_name_patterns = $cfg->get(
            'IMATCH_'.strtoupper($this->entity).'_NAME_PATTERNS',
            array('/^([^ ])$/', '/^([^ ])([^ ])$/', '/^([^ ])([^ ])([^ ])$/', '/^([^ ])([^ ])[^ ]* ([^ ]).*$/')
        );
    }
    
    public function deduceDtoIdStrFromName($name, $row_id=null)
    {
        if (empty($name)) return null;
        foreach ($this->dto_id_from_name_replacex as $search => $replacex) {
            $name = preg_replace($search, $replacex, $name);
        }
        if (empty($name)) return null;
        
        $matches = null;
        foreach ($this->dto_id_from_name_patterns as $pattern) {
            preg_match($pattern, $name, $matches);
            if ($matches) {
                $dto_id = strtoupper(implode('', array_slice($matches, 1)));
                if ($row_id) {
                    $other_row_id = $this->rowIdFromDtoId($dto_id);
                    if (!empty($other_row_id)) {
                        $this->task->log('WARN: '.$this->entity.' '.$row_id.' y '.$other_row_id.' comparten el mismo dto_id: '.$dto_id);
                    }
                }
                break;
            }
        }
        
        if (empty($dto_id)) {
            $msg = 'No se descartó ni se pudo deducir el dto_id para '.$this->entity.($row_id? '['.$row_id.']' : '').': '.$name;
            throw new Fs2psException($msg);
        }
        
        return $dto_id;
    }
    
    public function _dtoIdStrFromRowId($row_id)
    {
        if (empty($row_id)) return '';
        
        $dto_id = parent::_dtoIdStrFromRowId($row_id);
        if (!empty($dto_id)) return $dto_id;
        
        // Si no hay match trataremos de obtener el dtoId a partir del nombre
        $name = Fs2psTools::dbValue('
            select t.name from `@DB_terms` t
 			where t.term_id = '.$row_id.'
		');
        return $this->deduceDtoIdStrFromName($name, $row_id);
    }
}

class Fs2psSizeColourExtractorMatcher extends Fs2psDto2RowMatcher
{
    protected $productMatcher;
    protected $sizeMatcher;
    protected $colourMatcher;
    
    public $sizeAttributeGroupIds;
    public $sizeAttributeGroupIdsInSql;
    public $colourAttributeGroupIds;
    public $colourAttributeGroupIdsInSql;
    
    public function __construct($task, $entity) {
        $this->productMatcher = $task->getExtractor('products')->matcher;
        $this->sizeMatcher = $task->getExtractor('sizes')->matcher;
        $this->colourMatcher = $task->getExtractor('colours')->matcher;
        
        $attribute_group_matcher = $task->getExtractor('attribute_groups')->matcher;
        $this->sizeAttributeGroupIds = $attribute_group_matcher->rowIdFromDtoId('SIZES');
        $this->sizeAttributeGroupIdsInSql = $this->sizeAttributeGroupIds? (is_array($this->sizeAttributeGroupIds)? implode(',', $this->sizeAttributeGroupIds) : $this->sizeAttributeGroupIds) : 0;
        $this->colourAttributeGroupIds = $attribute_group_matcher->rowIdFromDtoId('COLOURS');
        $this->colourAttributeGroupIdsInSql = $this->colourAttributeGroupIds? (is_array($this->colourAttributeGroupIds)? implode(',', $this->colourAttributeGroupIds) : $this->colourAttributeGroupIds) : 0;
        
        parent::__construct($entity, 'product_attribute', 'id_product_attribute', null, null);
        $this->reloadCfg($task->cfg);
    }
    
    public function reloadCfg($cfg) {
        parent::reloadCfg($cfg);
        // USE_CREF indica que la combinación lleva su propia referencia.
        $this->dto_id_fields = $cfg->get('USE_CREF', False)? array('cref') : array('ref', 'size', 'colour');
        
        $pm = $cfg->get('COMBINATIONS_MATCHER');
        $this->persist = ($pm=='basic' || $pm=='pref');
    }
    
    public function _dtoIdStrFromRowId($row_id)
    {
        $dto_id = parent::_dtoIdStrFromRowId($row_id);
        if (!empty($dto_id)) return $dto_id;
        
        $r = Fs2psTools::dbRow('
			SELECT 
                max(pp.ID) as product,
                max(ts.term_id) as size,
                max(tc.term_id) as colour
			FROM
			    `@DB_posts` p
			    inner join `@DB_posts` pp on pp.ID=p.post_parent

                left join `@DB_woocommerce_attribute_taxonomies` wats on wats.attribute_id in ('.$this->sizeAttributeGroupIdsInSql.')
								-- Ejemplo evitar error diferentes encodings: convert(pacsv.meta_key USING utf8mb4)=convert(concat(\'attribute_pa_\', wats.attribute_name) USING utf8mb4)
                left join `@DB_postmeta` pacsv on pacsv.post_id=p.ID and pacsv.meta_key=concat(\'attribute_pa_\', wats.attribute_name)
                left join `@DB_term_taxonomy` tts on tts.taxonomy = concat(\'pa_\', wats.attribute_name)
				left join `@DB_terms` ts on ts.term_id=tts.term_id and ts.slug=pacsv.meta_value
                
				left join `@DB_woocommerce_attribute_taxonomies` watc on watc.attribute_id in ('.$this->colourAttributeGroupIdsInSql.')
                left join `@DB_postmeta` paccv on paccv.post_id=p.ID and paccv.meta_key=concat(\'attribute_pa_\', watc.attribute_name)
                left join `@DB_term_taxonomy` ttc on ttc.taxonomy = concat(\'pa_\', watc.attribute_name)
				left join `@DB_terms` tc on tc.term_id=ttc.term_id and tc.slug=paccv.meta_value
			WHERE p.ID='.$row_id.'
		    GROUP BY p.ID
		');
        if (!$r) return null;
        
        $product = $this->productMatcher->dtoIdStrFromRowId($r['product']);
        $size = $this->sizeMatcher->dtoIdStrFromRowId($r['size']);
        $colour = $this->colourMatcher->dtoIdStrFromRowId($r['colour']);
        
        if (!($product && ($size || $colour))) return null;
        
        return $this->dtoIdToStr(array($product, $size, $colour));
    }
}

class Fs2psOrderMatcher extends Fs2psDto2PostDirectMatcher {
    public function __construct($task, $entity) {
        parent::__construct($entity, array('ref'));
    }
}
