<?php
/**
 * LICENCIA
 *
 * Este programa se propociona "tal cual", sin garantía de ningún tipo más allá del soporte
 * pactado a la hora de adquirir el programa.
 *
 * En ningún caso los autores o titulares del copyright serán responsables de ninguna
 * reclamación, daños u otras responsabilidades, ya sea en un litigio, agravio o de otro
 * modo, que surja de o en conexión con el programa o el uso u otro tipo de acciones
 * realizadas con el programa.
 *
 * Este programa no puede modificarse ni distribuirse sin el consentimiento expreso del autor.
 *
 *    @author    Carlos Fillol Sendra <festeweb@festeweb.com>
 *    @copyright 2014 Fes-te web! - www.festeweb.com
 *    @license   http://www.festeweb.com/static/licenses/fs2ps_1.1.0.txt
 */

include_once(dirname(__FILE__).'/Fs2psTools.php');
include_once(dirname(__FILE__).'/Fs2psUpdater.php');
include_once(dirname(__FILE__).'/Fs2psObjectModels.php');


class Fs2psTermUpdater extends Fs2psUpdater {
	
	protected function dto2row($dto, $idx, $exists, $oldRowId) {
		$row = array();

		if ($this->enable) {
			$row['is_visible'] = 1;
		} else if (!$exists) {
			$row['is_visible'] = 0;
		}

		if (!$exists || !$this->noover_content) {
			if (isset($dto['name']))
			{
				$name = $dto['name'];
				$row['name'] = $name;
				if (!empty($name) && empty($dto['slug']) && (!$exists || !$this->noover_url)) {
					$row['slug'] = sanitize_title($name);
				}
			}

			if (!empty($dto['slug']) && (!$exists || !$this->noover_url)) {
				$row['slug'] = sanitize_title($dto['slug']);
            }

			if (isset($dto['descrip'])) {
				$row['description'] = $dto['descrip'];
			}
		}
		return $row;
	}
	
	public function process($dtos) {
	    // No hacemos nada sin una taxonomía válida
	    if (empty($this->matcher->taxonomy)) return;
	    
	    parent::process($dtos);
	}
	
	protected function insertOrUpdate($row, $exists, $oldRowId) {
		$result = null;
		
		if ($exists) {
			$result = Fs2PsTools::wpErr(wp_update_term($oldRowId, $this->matcher->taxonomy, $row));
		} else {
			$result = Fs2PsTools::wpErr(
			    wp_insert_term($row['name'], $this->matcher->taxonomy, $row));
		}
		
		return $result['term_id'];
	}
	
}

class Fs2psCategoryUpdater extends Fs2psTermUpdater {
	
	protected function reloadCfg() {
		parent::reloadCfg();
		$cfg = $this->task->cfg;
		$this->noover_parent = $cfg->get('NOOVER_CATEGORIES_PARENT', false);
	}
	
	protected function dto2row($dto, $idx, $exists, $oldRowId) {
		$row = parent::dto2row($dto, $idx, $exists, $oldRowId);
		if (!$exists || !$this->noover_parent) {
			$row['parent'] = !empty($dto['parent'])? $this->matcher->rowIdFromDtoId($dto['parent']) : '';
		}
		return $row;
	}
	
}

class Fs2psManufacturerUpdater extends Fs2psTermUpdater { }

class Fs2psSupplierUpdater extends Fs2psTermUpdater { }

class Fs2psGroupUpdater extends Fs2psUpdater
{
    protected $DEFAULT_CAPABILITIES = array( 'read' => true, );
    
    protected function dto2row($dto, $idx, $exists, $oldRowId)
    {
        $row = array(
            'role' => $this->matcher->rowIdFromDto($dto)
        );
        if (!$exists || !$this->noover_content)
        {
            if (isset($dto['name']))  $row['name'] = $dto['name'];
        }
        return $row;
    }
    
    protected function insertOrUpdate($row, $exists, $oldRowId, $oldObj=null)
    {
        if (!$exists) {
            add_role($row['role'], $row['name'], $this->DEFAULT_CAPABILITIES);
        } else if (!empty($row['name'])) {
            
            global $wpdb;
            $user_roles_prop = $wpdb->prefix.'user_roles';
            
            $val = get_option($user_roles_prop);
            $val[$row['role']]['name'] = $row['name'];
            update_option($user_roles_prop, $val );
            
            /* TODO: Cambiar nombre
            global $wp_roles;
            
            if (!isset($wp_roles))
                $wp_roles = new WP_Roles();
                
                $role =& get_role($role_name);
                
                $wp_roles->roles[$role]['name'] = $new_role_name; //not working for sa
                $wp_roles->role_names[$role] = $display_name;       
           */  
        }
        return $row['role'];
    }
}

class Fs2psAttributeGroupUpdater extends Fs2psUpdater {
	
	protected function dto2row($dto, $idx, $exists, $oldRowId) {
	    
		$row = array(
			'attribute_type' => 'select',
			'attribute_orderby' => 'menu_order'
		);
		
		if ($this->enable) {
			$row['attribute_public'] = 1;
		} else if (!$exists) {
			$row['attribute_public'] = 0;
		}
		
		if (!$exists) {
		    $matcher = $this->matcher;
		    $row['attribute_name'] = $matcher->attributeNameFromDtoId($matcher->dtoId($dto));
		}
		if (!$exists || !$this->noover_content) {
	        if (isset($dto['name'])) {
		      $row['attribute_label'] = $dto['name'];
		    }
		}
		
		return $row;
	}
	
	protected function insertOrUpdate($row, $exists, $oldRowId) {
        global $wpdb;
        if ($exists) {
            $id = $oldRowId;
            $wpdb->update($wpdb->prefix.'woocommerce_attribute_taxonomies', $row, array('attribute_id'=>$id));
        } else {
            // If the attribute does not exist, create it.
            $id = Fs2PsTools::wpErr(wc_create_attribute(
                array(
                    'name'         => $row['attribute_label'],
                    'slug'         => $row['attribute_name'],
                    'type'         => $row['attribute_type'],
                    'order_by'     => $row['attribute_orderby'],
                    'has_archives' => false,
                )
            ));
            
            // Register as taxonomy
            //
            // IMPORTANTE:
            // Las taxonomías no están disponibles porque WP las hace disponibles en el init. Estarán disponibles en la próxima request.
            // Por eso hay que crear grupos de atributos y dar valores a esos atributos en dos requests distintas.
            $tax_name = 'pa_'.$row['attribute_name'];
            register_taxonomy(
                $tax_name,
                apply_filters('woocommerce_taxonomy_objects_' .$tax_name, array('product') ),
                apply_filters('woocommerce_taxonomy_args_' . $tax_name, array(
                    'labels'       => array( 'name' => $row['attribute_label'], ),
                    'hierarchical' => true,
                    'show_ui'      => false,
                    'query_var'    => true,
                    'rewrite'      => false,
                ))
            );
          }
            
          return $id;
	}
}

class Fs2psAttributeUpdater extends Fs2psTermUpdater { }

class Fs2psSKUUpdater extends Fs2psUpdater {
    
	protected $trigger_WC_hooks;
    protected $nogroup_specific_prices;
    protected $prices_include_tax;
    protected $tax_class_by_rate;
    protected $noover_sale_price_dates;
	protected $noover_producttype;
    protected $woo3ormore;
    protected $rbp_activated;
    protected $b2b_activated;
    protected $cheap_b2b_activated;
    protected $b2b_king_activated;
	protected $b2b_king_page_cache = array();
    protected $yith_activated;
    protected $groupMatcher;
    protected $manage_stock;
    protected $manage_stock_ifnotexist;
    protected $allow_zero_prices;
    protected $managing_price_from_quantity;
	protected $num_decimals;
	protected $update_stockables_combis_can_be_products;
    
	protected $dims = array('width'=>'width', 'height'=>'height', 'depth'=>'length', 'weight'=>'weight');

    protected $woo3prop_by_key = array(
        'ID' => 'id',
        'post_parent' => 'parent_id',
        'post_title' => 'name',
        'post_excerpt' => 'short_description',
        'post_content' => 'description',
        '_visibility' => 'catalog_visibility',
        '_sale_price_dates_to' => 'date_on_sale_to',
        '_sale_price_dates_from' => 'date_on_sale_from',
    );
    
    protected function reloadCfg() {
        parent::reloadCfg();
        $cfg = $this->task->cfg;
        $this->nogroup_specific_prices = $cfg->get('NOGROUP_SPECIFIC_PRICES', FALSE);
        $this->prices_include_tax = $cfg->get('PRICES_INCLUDE_TAX');
        $this->noover_sale_price_dates = $cfg->get('NOOVER_SALE_PRICE_DATES', FALSE);
		$this->noover_producttype = $cfg->get('NOOVER_PRODUCTS_PRODUCTTYPE', false);
        $this->woo3ormore = Fs2psTools::wooVer('3');
        $this->managing_price_from_quantity = $cfg->get('MANAGING_PRICE_FROM_QUANTITY', FALSE);
        $this->rbp_activated = function_exists('wc_rbp_update_role_based_price');
        $this->b2b_activated = class_exists('Addify_B2B_Plugin');
        $this->b2b_king_activated = class_exists('B2bking') || class_exists('B2bkingcore');
		$this->b2b_wholesale = Fs2psTools::isPluginActive('woocommerce-wholesale-pro/woocommerce-wholesale-pro.php');
        $this->cheap_b2b_activated = class_exists( 'Addify_Customer_And_Role_Pricing' );
        $this->yith_activated = function_exists( 'yith_role_based_prices_premium_init' );
        $this->groupMatcher = Fs2psMatcherFactory::get($this->task, 'price_rates');
        $this->kses_remove_filters = $cfg->get('KSES_REMOVE_FILTERS', TRUE);
		$this->update_stockables_combis_can_be_products = $cfg->get('UPDATE_STOCKABLES_COMBIS_CAN_BE_PRODUCTS', False);
        
        $this->round_saved_price = $cfg->get('ROUND_SAVED_PRICE', FALSE);
        
        $manage_stock = $cfg->get('MANAGE_STOCK', [  'true'  ]);
        $manage_stock_opts = array_map('strtolower', $manage_stock);
        $this->manage_stock = in_array('false', $manage_stock_opts) ? FALSE : TRUE;
        $this->manage_stock_ifnotexist = in_array('ifnotexist', $manage_stock_opts);
		$this->notify_no_stock_amount = get_option('woocommerce_notify_no_stock_amount'); // TODO Moisés: Obtener de wp_options
        
        $this->allow_zero_prices = $cfg->get('ALLOW_ZERO_PRICE', false);
        
        $this->price_num_decimals = Fs2psTools::dbValue('select option_value from @DB_options where option_name="woocommerce_price_num_decimals"');
        
		$this->wp_lister_active = class_exists( 'WPL_WPLister' );
		$this->trigger_WC_hooks = $cfg->get("TRIGGER_WC_HOOKS", false);
		$this->b2b_replace_original_price = $cfg->get('B2B_REPLACE_ORIGINAL_PRICE_DEFAULT', TRUE);
    }
    
    protected function saving_price($price){
        $rsp = $this->round_saved_price;
        return $rsp!==false ? round($price, $rsp===true? $this->price_num_decimals : $rsp) : $price;
    }

	protected function formated_saving_price($price){
		$rsp = $this->round_saved_price;
		if ($rsp===false) return str_replace('.', ',', $price);
		$ndecimals = $rsp===true? $this->price_num_decimals : $rsp;
		return number_format($this->saving_price($price), $ndecimals, ',', ' ' ); 
    }

	protected function hasToOverrideProductType($post_id) {
		if (!$this->noover_producttype) return true;
		if ($this->noover_producttype===true) return false;
		$existing_product_types = Fs2PsTools::wpErr(wp_get_object_terms($post_id, 'product_type'));
		$noover = $this->noover_producttype;
		if (array_filter($existing_product_types, function ($e) use ($noover) { return $e->slug==$noover; })) return false;
		return true;
	}

    protected function loadTaxRulesGroupByRate() {
        $tax_rules = Fs2psTools::dbSelect('
			SELECT distinct tr.tax_rate_class, tr.tax_rate
			FROM @DB_woocommerce_tax_rates tr
		');
        $tax_class_by_rate = array();
        foreach ($tax_rules as $rule)
        {
            $rate = (string)((float)$rule['tax_rate']);
            $tax_class_by_rate[$rate] = $rule['tax_rate_class'];
        }
        $this->tax_class_by_rate = $tax_class_by_rate;
    }
    
	protected function dto2row($dto, $idx, $exists, $oldRowId) {
	    
		$row = array(
		    '_sku' => $this->matcher->referenceFromDto($dto),
		);
		
		// ¿Se gestionan existencias en este producto/variante?
		if ($exists && $this->manage_stock_ifnotexist) {
		    $row['_manage_stock'] = get_post_meta($oldRowId, '_manage_stock', $this->manage_stock? 'yes' : 'no');
		} else { // if (!$exists || !$this->manage_stock_ifnotexist) 
		    $row['_manage_stock'] = $this->manage_stock? 'yes' : 'no';
		}

		$manage_stock = $row['_manage_stock']=='yes';
		
		if (isset($dto['stock']) || !$exists) {
		    $row['_stock'] = empty($dto['stock'])? 0 : $dto['stock'];
		}
		
		if (isset($dto['backorders'])) { // backorders = allow/deny/notify
		    $row['_backorders'] = $dto['backorders']=='allow'? 'yes' : ($dto['backorders']=='deny'? 'no' : 'notify');
		}
		if ($manage_stock && $exists && isset($row['_stock']) && $row['_stock']<=$this->notify_no_stock_amount) {
		    // Tomamos nota de backorders si el stockable existe y el stock es negativo. Su estado de existencias dependerá del valor de backorders.
		    $backorders = empty($row['_backorders'])? get_post_meta($oldRowId, '_backorders', TRUE) : $row['_backorders'];
	        $row['__backorders'] = $backorders;
		}
		
		if (isset($dto['prices']) || isset($dto['price'])) {
		    $prices = isset($dto['prices']) ? $dto['prices'] : array( array('price' => $dto['price']) );
		    $row['__prices'] = $prices;
		    $price = $prices[0];
		    
	        $rate = $dto['iva'];
	        $row['__rate'] = $rate;
	        $regular_price = $price['price']; 
	        
	        if ($regular_price<=0 && $this->allow_zero_prices || $regular_price>0) {
	            if ($this->prices_include_tax) $regular_price = floatval(($regular_price * (1+$rate)) * 100)/100.0;
	            $row['__regular_price'] = $this->saving_price($regular_price);
	        }
	        
	        if ($this->nogroup_specific_prices) {
	            // Trabajamos con descuentos sólo si los gestionamos según la configuración
	            $discount_field = isset($price['disp'])? 'disp' : 'dis';
	            $discount = empty($price[$discount_field])? 0.0 : $price[$discount_field]; // Siempre sin IVA
	            if ($discount_field=='disp') $sale_price = $regular_price * (100 - $discount)/100.0;
	            else {
	                if ($discount && $this->prices_include_tax) $discount = floatval(($discount * (1+$rate)) * 100)/100.0;
	                $sale_price = $regular_price - $discount;
	            }
	            if ($discount && ($sale_price<=0 && $this->allow_zero_prices || $sale_price>0)) {
	                $row['__sale_price'] = $this->saving_price($sale_price);
	            }
	            
	            if (!$this->noover_sale_price_dates) { // Gestionamos fechas?
	                $row['_sale_price_dates_from'] = $discount && !empty($price['dis_from'])? strtotime($price['dis_from']) : '';
	                $row['_sale_price_dates_to'] = $discount && !empty($price['dis_to'])? strtotime($price['dis_to']) : '';
	            }
	        }
	        
		}
		
		if (!empty($dto['custom_fields'])) $row['__custom_fields'] = $dto['custom_fields'];
		if (!empty($dto['custom_attributes'])) $row['__custom_attributes'] = $dto['custom_attributes'];
		
		// Peso y Medidas
		foreach ($this->dims as $dto_dim=>$woo_dim) {
		    if (isset($dto[$dto_dim])) 
			    $row['_'.$woo_dim] = $dto[$dto_dim];
		}

		if (isset($dto['shipping_class'])) $row['__shipping_class'] = $dto['shipping_class'];

		if (isset($dto['shipping_class_slug'])) $row['__shipping_class_slug'] = $dto['shipping_class_slug'];

		return $row;
	}

	protected function insertOrUpdate($row, $exists, $oldRowId) {
	    
	    $id = null;
	    
	    if ($this->kses_remove_filters) kses_remove_filters(); // TODO Mover a onProcessStart
	        
		$post = array();
		foreach ($row as $key => $value) {
		    if (preg_match("/^post_.*/", $key)) {
				$post[$key] = $value;
		    }
		}
				
		$old_post = $exists? get_post($oldRowId) : null;
		if ($old_post!==null) {
			$post['ID'] = $oldRowId;
			// Solo actualizamos si hay cambios. wp_update_post es muy costoso. Ver FS2PS-782
			foreach ($post as $key => $value) {
				if ($old_post->$key != $value) {
					Fs2PsTools::wpErr(wp_update_post($post, true));
					break;
				}
			}
			
			$id = $oldRowId;
		} else {
			if (($this->matcher->direct_match || $this->matcher instanceof Fs2psDto2RowDirectMatcher) && !empty($oldRowId)) {
				$post['import_id'] = $oldRowId;
			}
			$id = Fs2PsTools::wpErr(wp_insert_post($post, true));
		}

		foreach ($row as $key => $value) {
			if (preg_match("/^_[^_].*/", $key)) {
				update_post_meta( $id, $key, $value );
			}
		}

		//Compatibilidad con WP_Lister. Notificamos al plugin que ha cambiado el stock.
		if ($this->wp_lister_active && !empty($row['_stock'])) do_action('wplister_product_has_changed', $id);
		
		$manage_stock = $row['_manage_stock']=='yes';
		if (isset($row['_stock']) && $manage_stock) {
		    $stock_status = $row['_stock']>$this->notify_no_stock_amount? 'instock' : (empty($row['__backorders']) || $row['__backorders']=='no'? 'outofstock' : 'onbackorder');
		    update_post_meta($id, '_stock_status', $stock_status);
		    
		    // A partir de WooCommerce 3 hay que actualizar también la taxonomia 
		    if ($stock_status=='outofstock') Fs2PsTools::wpErr(wp_set_post_terms($id, 'outofstock', 'product_visibility', true));
		    else Fs2PsTools::wpErr(wp_remove_object_terms($id, 'outofstock', 'product_visibility'));
		} else if (!$manage_stock && (!$exists || !$this->manage_stock_ifnotexist) ) {
		    update_post_meta($id, '_stock_status', 'instock');
		    Fs2PsTools::wpErr(wp_remove_object_terms($id, 'outofstock', 'product_visibility'));
		}
		
		if (isset($row['__prices'])) {
		    if (!isset($row['__regular_price'])) {
    			// Producto/Combinación sin precio
    			# '_sale_price_dates_from','_sale_price_dates_to' no los borramos para respetar config. Los pondremos a '' si cabe.
    		    delete_post_meta($id, '_regular_price');
    			update_post_meta($id, '_price', '');
    			if ($this->nogroup_specific_prices) delete_post_meta($id, '_sale_price');
    		} else {
    			// Producto/Combinación con precio y con posible oferta
    		    update_post_meta($id, '_regular_price', $row['__regular_price']);
    		    $sale_price = $this->nogroup_specific_prices? (isset($row['__sale_price'])? $row['__sale_price'] : false) : ($exists? get_post_meta($id, '_sale_price', true) : false);
    		    $price = empty($sale_price) || $sale_price>$row['__regular_price']? $row['__regular_price'] : $sale_price;
    			update_post_meta($id, '_price', $price);
    			if ($this->nogroup_specific_prices) {
    			    !isset($row['__sale_price'])? delete_post_meta($id, '_sale_price') : update_post_meta($id, '_sale_price', $price);
    			}
    		}
    		$this->setPricesByRole($row, $id);
    		
		}

		if (isset($row['__custom_fields'])) {
		    foreach ($row['__custom_fields'] as $k => $v) update_post_meta($id, $k, $v);
		}

		if (isset($row['__custom_attributes'])) {
			$data = get_post_meta($id, '_product_attributes', true);
			if (empty($data)||!is_array($data)) $data = array();
		    foreach ($row['__custom_attributes'] as $k => $v) {
				$pa = 'pa_'.$k;
				if ((empty($v) && $v!=='0')) {
					if (isset($data[$pa])) {
						Fs2PsTools::wpErr(wp_set_object_terms($id, NULL, $pa, false));
						unset($data[$pa]);
					}
				} else {
					Fs2PsTools::wpErr(wp_set_object_terms($id, $v, $pa, false));
					if (empty($data[$pa])) {
						$data[$pa] = array(
							'name' => $pa,
							'value' => '',
							'is_visible' => '1',
							'is_taxonomy' => '1'
						);
					}
				}
			}
			update_post_meta($id, '_product_attributes', $data);
		}

		if (isset($row['__shipping_class'])) {
		    $shipping_classes = empty($row['__shipping_class'])? array() : array((int)$row['__shipping_class']);
		    Fs2PsTools::wpErr(wp_set_post_terms($id, $shipping_classes, 'product_shipping_class'));
		}

		if (isset($row['__shipping_class_slug'])) {
			Fs2PsTools::wpErr(wp_set_post_terms($id, $row['__shipping_class_slug'], 'product_shipping_class'));
		}
		
		if ($this->kses_remove_filters) kses_init_filters(); // TODO Mover a onProcessEnd

		//Disparamos los hooks.
		if ($this->trigger_WC_hooks == true) {
			$_product = wc_get_product( $id );
			if ($_product->post_type == 'product') {
				do_action( 'woocommerce_update_product', $id, $_product );
				do_action( 'woocommerce_product_set_stock', $_product );
				do_action( 'woocommerce_product_set_stock_status', $id, $_product->get_stock_status(), $_product );
			} else {
				do_action( 'woocommerce_update_product_variation', $id, $_product );
				do_action( 'woocommerce_variation_set_stock', $_product );
				do_action( 'woocommerce_variation_set_stock_status', $id, $_product->get_stock_status(), $_product );
			}
		}

	    return $id;
	}
	
	/**
	 * @param mixed &$regular_price Parámetro de salida. Precio regular.
	 * @param mixed &$sale_price Parámetro de salida. Precio de oferta. Si hay oferta $sale_price!==null && $sale_price<$regular_price.
	 * @param mixed $row Array de precios.
	 * @param mixed $price Precio del grupo actual
	 * @return Nada. La respuesta se da en los parámetros de salida regular_price y sale_price.
	 */
	protected function getRegularAndSaleFromPrice(&$regular_price, &$sale_price, $row, $price)
	{
	    $regular_price = $price['price'];
	    if ($this->prices_include_tax){
	        $regular_price = floatval(($regular_price * (1+$row['__rate'])) * 100)/100.0;
	    }
	    //$sale_price = null;
	    if ($this->nogroup_specific_prices) { // Gestionamos ofertas
	        $discount_field = isset($price['disp'])? 'disp' : 'dis';
	        $discount = empty($price[$discount_field])? 0.0 : $price[$discount_field]; // Siempre sin IVA
	        if ($discount_field=='disp') $sale_price = $regular_price * (100 - $discount)/100.0;
	        else {
	            if ($discount && $this->prices_include_tax) $discount = intval(($discount * (1+$row['__rate']) + 0.005) * 100)/100.0; // Redondea descuento ¿Donde?
	            $sale_price = $regular_price - $discount;
	        }
	    }
	}

	protected function setPricesByRole($row, $id)
	{

	    if (empty($row['__prices']) || sizeof($row['__prices'])<=1) return; // No gestionamos multitarifa
	    
	    if ($this->rbp_activated || $this->b2b_activated || $this->yith_activated || $this->cheap_b2b_activated || $this->b2b_king_activated || $this->b2b_wholesale) {
	        // Comprobaciones genericas iniciales sobre datos que llegan
	        if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
	            $prices = $row['__prices'];
	            foreach($prices as $idx=>$price)
	            {
	                if ($idx==0) continue; // Precio por defecto
	                
	                if (empty($price['group']) && empty($price['fromqty'])) {
	                    $msg = 'WARN: No se indicó grupo o cantidad mínima en precio específico';
	                    $this->task->log($msg);
	                    continue;
	                }
	                
	                if (!empty($price['customer'])) {
	                    $msg = 'WARN: Actualmente no se soportan descuentos por cliente ('.$price['customer'].')';
	                    $this->task->log($msg);
	                    continue;
	                }
	                
	                if (!empty($price['group'])) {
    	                $role = $this->groupMatcher->rowIdFromDtoId($price['group']);
    	                if (empty($role)) {
    	                    $msg = 'WARN: El grupo "'.$price['group'].'" no existe';
    	                    $this->task->log($msg);
    	                    continue;
    	                }
	                }
	            }
	        }
	    } 
	        
	    if ($this->rbp_activated) {
    	    $rbp_prices = array();
    	    
    	    if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
                $prices = $row['__prices'];
                foreach($prices as $idx=>$price)
                {
                    if ($idx==0) continue; // Precio por defecto
                    
                    $role = $this->groupMatcher->rowIdFromDtoId($price['group']);
                    
                    $regular_price = $sale_price = null;
                    $this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);

					$rbp_prices[$role]['regular_price'] = $this->formated_saving_price($regular_price); // De este modo se pone la , y no .
                    //$rbp_prices[$role]['regular_price'] = wc_format_decimal($this->saving_price($regular_price)); // wc_format_decimal? Sigue saliendo . en masquecamper en lugar de ,
                    if ($sale_price<=$regular_price && !empty($sale_price)) {
                        //$rbp_prices[$role]['selling_price'] = wc_format_decimal($this->saving_price($sale_price));
						$rbp_prices[$role]['selling_price'] = $this->formated_saving_price($sale_price);
					}
                }
    	    }
            
    	    if (!empty($rbp_prices)) {
    	        wc_rbp_update_role_based_price_status($id, TRUE);
    	        wc_rbp_update_role_based_price($id, $rbp_prices, FALSE);
    	    } else {
    	        // TODO: No se llega nunca a ejecutar. Esta OK?
    	        delete_post_meta($id, '_role_based_price');
    	        delete_post_meta($id, '_enable_role_based_price');
    	    }
	    
	    }
	    
	    if ($this->b2b_activated) {
	        $b2b_prices = array();
	        
	        if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
	            $prices = $row['__prices'];
	            
	            $fromqty_discounts = [];
	            $secure_insert = false; 
	            
	            foreach($prices as $idx=>$price)
	            {
	                //OUTPUT EXEMPLE
	                /*[[0] => [[price] => 40, [group] => 1], 
	                 * [1] => [[disp] => 5.0000, [price] => 40, [fromqty] => 20.0000, [dis] => 38], 
	                 * [2] => [[price] => 200, [group] => 2], 
	                 * [3] => [[disp] => 5.0000, [price] => 200, [fromqty] => 20.0000, [group] => 2, [dis] => 190], 
	                 * [4] => [[price] => 10, [group] => 3], 
	                 * [5] => [[disp] => 5.0000, [price] => 10, [fromqty] => 20.0000, [group] => 3, [dis] => 9.5]]*/	                
	                
	                if ($idx==0) continue; // Precio por defecto
	                if (!$this->managing_price_from_quantity && array_key_exists('fromqty', $price)) continue;
	                
	                $role = $this->groupMatcher->rowIdFromDtoId($price['group']);
	                if (empty($role)){
	                    $role = $this->groupMatcher->rowIdFromDtoId(1);
	                }
	                
	                if (array_key_exists('fromqty', $price)){
	                    
	                    
	                    if(is_array($prices[$idx+1]) && array_key_exists('fromqty', $prices[$idx+1])){
	                        $secure_insert = false;
	                    }else{
	                        $secure_insert = true;
	                        foreach($fromqty_discounts as $idx_two=>$array){
	                            $array['max_qty'] = intval($price['fromqty']) -($idx_two+1);
	                            array_push($b2b_prices, $array);
	                        }
	                    } 
	                    
	                    $b2b_idx_array = array(
	                        'user_role' => '',
	                        'discount_type' => 'fixed_price',
	                        'discount_value' => '',
	                        'min_qty' => '',
	                        'max_qty' => '',
	                    );
	                    $b2b_idx_array['user_role'] = $role;
	                    $b2b_idx_array['min_qty'] = intval($price['fromqty']);
						if(empty($price['dis']) && !empty($price['disp'])){
							$price['dis'] = $price['price']* ((100 - $price['disp'])/100.0);
						} 
	                    $b2b_idx_array['discount_value'] = $price['dis'];
	                    if ($this->prices_include_tax){
	                        $b2b_idx_array['discount_value'] = $this->saving_price(floatval(($price['dis'] * (1+$row['__rate'])) * 100)/100.0);
	                    }
	                    
	                    $fromqty_discounts[] = $b2b_idx_array;
	                    
	                }else{
	                    
	                    $fromqty_discounts = [];
	                    $secure_insert = true;
	                    
	                    $b2b_idx_array = array(
	                        'user_role' => '',
	                        'discount_type' => 'fixed_price',
	                        'discount_value' => '',
	                        'min_qty' => '1',
	                        'max_qty' => '',
	                        'replace_orignal_price' => $this->b2b_replace_original_price ? 'yes' : 'no',
	                    );
	                    
	                    if(is_array($prices[$idx+1]) && array_key_exists('fromqty', $prices[$idx+1])){
	                        $b2b_idx_array['max_qty'] = intval($prices[$idx+1]['fromqty']) -1;
	                    }
	                    
	                    $regular_price = $sale_price = null;
	                    $this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);

						if (empty($sale_price)) $sale_price = $regular_price;
	                    
	                    if ($sale_price<$regular_price) {
	                        //sale_price y regular_price valen lo mismo??
	                        unset($b2b_idx_array['replace_orignal_price']);
	                    }
	                    $b2b_idx_array['discount_value'] = $this->saving_price($sale_price);
	                    $b2b_idx_array['user_role'] = $role;
	                }
	                
	                if($secure_insert != false){
	                    array_push($b2b_prices, $b2b_idx_array);
	                }
	            }
	        }
	        
	        if (!empty($b2b_prices)) {
	            update_post_meta($id, '_role_base_price', $b2b_prices);
	        } else {
	            // TODO: No se llega nunca a ejecutar. Esta OK?
	            delete_post_meta($id, '_role_base_price');
	        } 
	    }

		if ($this->yith_activated) {
	        $yith_prices = array();
			$first_sale_price = null;
	        
	        if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
	            $prices = $row['__prices'];
	            foreach($prices as $idx=>$price)
	            {
	                
	                if ($idx==0) {
						// Precio por defecto
						$regular_price = $sale_price = null;
	                    $this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);
						$first_sale_price = $sale_price;
						continue; 
					}
	                
	                $yith_idx_array =   array (
	                    'rule_name' => '',
	                    'rule_role' => '',
	                    'rule_type' => 'discount_val',
	                    'rule_value' => '',
	                );
	                
	                $role = $this->groupMatcher->rowIdFromDtoId($price['group']);

	                $regular_price = $sale_price = null;
	                $this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);
	                
	                $yith_idx_array['rule_role'] = $role;
	                $yith_idx_array['rule_name'] = $role;
					/* Como hacemos esto para que el cliente vea los precios segun las tarifas de Factusol?? */
	                $yith_idx_array['rule_value'] = $this->saving_price($first_sale_price - $sale_price);
					//$yith_idx_array['rule_value'] = $this->saving_price($sale_price);
	                
	                array_push($yith_prices, $yith_idx_array);
	            }
	        }
	        
	        if (!empty($yith_prices)) {
	            update_post_meta($id, '_product_rules', $yith_prices);
	        } else {
	            // TODO: No se llega nunca a ejecutar. Esta OK?
	            delete_post_meta($id, '_product_rules');
	        }
	    }
	    
	    if ($this->cheap_b2b_activated) {
	        $b2b_prices = array();
	        
	        if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
	            $prices = $row['__prices'];
	            //$role_rows = [];
	            foreach($prices as $idx=>$price)
	            {
	                
	                if ($idx==0) continue; // Precio por defecto
	                
	                $b2b_idx_array = array(
	                    'discount_type' => 'fixed_price',
	                    'discount_value' => '',
	                    'min_qty' => '1',
	                    'max_qty' => '',
	                    'replace_orignal_price' => $this->b2b_replace_original_price ? 'yes' : 'no',
	                );
	                
	                $role = $this->groupMatcher->rowIdFromDtoId($price['group']);
	                
	                $regular_price = $sale_price = null;
	                $this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);
	                if ($sale_price!==null && $sale_price<$regular_price) {
	                    unset($b2b_idx_array['replace_orignal_price']);
	                } // else $sale_price>=$regular_price
	                
	                $b2b_idx_array['discount_value'] = $this->saving_price($sale_price);
	                $row_name = '_role_base_price_' . trim(strval($role));
	                $b2b_prices[$row_name] = $b2b_idx_array;
	            }
	        }
	        
	        if (!empty($b2b_prices)) {
	            
	            foreach($b2b_prices as $row_name => $row_value){
	                update_post_meta($id, $row_name, serialize($row_value));
	            }   
	            
	        } else {
	            // TODO: No se llega nunca a ejecutar. Esta OK?
	            foreach($b2b_prices as $row_name => $row_value){
	                delete_post_meta($id, $row_name);
	            }   
	            
	        }
	    }
	    
	    if ($this->b2b_king_activated) {
	        
	        if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
	            
                $prices = $row['__prices'];
                $fromqty_discounts = array();
    
                $group = null;
                $next_group = null;
                $last_idx = count($prices)-1;

                foreach($prices as $idx=>$price)
                {
                    //OUTPUT EXEMPLE
                    /*[[0] => [[price] => 40, [group] => 1],
                     * [1] => [[disp] => 5.0000, [price] => 40, [fromqty] => 20.0000, [dis] => 38],
                     * [2] => [[price] => 200, [group] => 2],
                     * [3] => [[disp] => 5.0000, [price] => 200, [fromqty] => 20.0000, [group] => 2, [dis] => 190],
                     * [4] => [[price] => 10, [group] => 3],
                     * [5] => [[disp] => 5.0000, [price] => 10, [fromqty] => 20.0000, [group] => 3, [dis] => 9.5]]*/
                    
                    $group = empty($price['group'])? null : $price['group'];
                    $next_group = $idx<$last_idx && array_key_exists('group', $prices[$idx+1])? $prices[$idx+1]['group'] : null;
                    
                    if (array_key_exists('fromqty', $price)){
                        $fromqty_discounts[$price['fromqty']] = $price['dis'];
                    }else{
                        //Borramos si el producto no tiene precio por cantidad
                        if($next_group!=null && $next_group!=$group){
                            $role = $this->groupMatcher->rowIdFromDtoId($group); 
                            if($idx == 0){
                                delete_post_meta($id,'b2bking_product_pricetiers_group_b2c');
                            }
                        }
                    }
                    if ($idx==0) continue; // Precio por defecto
                    
                    if ($group!=$next_group || $idx==$last_idx) {
                        // Llegamos al final del grupo y guardamos información recogida
                        
                        $pricetier_string = $this->manage_b2bking_pricetier($fromqty_discounts);
                        
                        if ($group) {
                            $role = $this->groupMatcher->rowIdFromDtoId($price['group']);
							#Se crean los grupos otra vez al llamar este metodo..? Deberiamos de depurar
                            $page_id = $this->getB2bKingPage($role);
                            
                            $regular_price = $sale_price = null;
                            $this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);
                            
                            update_post_meta($id, 'b2bking_regular_product_price_group_'.$page_id, $regular_price);
                            if ($this->managing_price_from_quantity){
                                update_post_meta($id, 'b2bking_product_pricetiers_group_'.$page_id, $pricetier_string);
                            }
                        } else {
                            // Regular price de precio por defecto ya establecido
                            // Establecemos sólo descuentos por cantidad para todos
                            //if (!$this->managing_price_from_quantity) continue;
                            if ($this->managing_price_from_quantity){
                                update_post_meta($id, 'b2bking_product_pricetiers_group_b2c', $pricetier_string);
                            }
                        }
                        // Inicializamos estado nuevo grupo
                        $fromqty_discounts = [];
                    }
                }
	        }
	    }
		if ($this->b2b_wholesale) {
	        if (empty($row['__combinations'])) { // Optimización: Si tiene combis podremos precios por role del padre a 0
                $prices = $row['__prices'];
    
                $group = null;
                $next_group = null;
                $last_idx = count($prices)-1;

                foreach($prices as $idx=>$price)
                {
                    $group = empty($price['group'])? null : $price['group'];
                    $next_group = $idx<$last_idx && array_key_exists('group', $prices[$idx+1])? $prices[$idx+1]['group'] : null;
                    
                    if ($idx==0) continue; // Precio por defecto
                    
                    if ($group!=$next_group || $idx==$last_idx) {
						$role = $this->groupMatcher->rowIdFromDtoId($price['group']);

						$regular_price = $sale_price = null;
						$this->getRegularAndSaleFromPrice($regular_price, $sale_price, $row, $price);
						
						//Guardamo en el meta
						if(!empty($sale_price) && $sale_price<$regular_price) {
							update_post_meta($id, $role, $sale_price);
						} else {
							update_post_meta($id, $role, $regular_price);
						}
                    }
                }
	        }
	    }
	}
	
	function manage_b2bking_pricetier($pricetier, $action = 'serialize'){
	    if($action == "serialize"){
	        $pricetier_string = "";
	        foreach($pricetier as $quantity=>$dis){
	            $pricetier_string .= $quantity.':'.$dis.';';
	        }
	        return $pricetier_string;
	    }
	}
	
	protected function getB2bKingPage($role) {

		if (!empty($this->b2b_king_page_cache[$role])) {
			return $this->b2b_king_page_cache[$role];
		}
		
		$page = get_page_by_path( html_entity_decode( $role ), OBJECT, 'b2bking_group');
		if(empty($page)) {
	        // Create post object
	        $group_post = array(
	            'post_name'    => wp_strip_all_tags($role),
				'post_title'    => wp_strip_all_tags($role),
	            'post_content'  => '',
	            'post_status'   => 'publish',
	            'post_author'   => 1,
	            'post_type' => 'b2bking_group'
	        );
	        return wp_insert_post( $group_post );
	    }

		$this->b2b_king_page_cache[$role] = $page->ID;

	    return $page->ID;
	}
}	

class Fs2psProductUpdater extends Fs2psSKUUpdater {
	
	protected $categories_upd;
	protected $manufacturers_upd;
	protected $suppliers_upd;
	protected $reltaxos;

	protected $noover_categories;
	protected $noover_manufacturer;
	
	protected $noover_name;
	protected $noover_descrip;
	protected $noover_longdescrip;
	protected $noover_products_tags;
	protected $post_author;
	
	public function __construct($task, $name) {
	    parent::__construct($task, $name);
		$this->categories_upd = $task->getUpdater('categories');

		$this->reltaxos = ['manufacturers', 'suppliers'];
		$this->manufacturers_upd = $task->getUpdater('manufacturers');
		$this->suppliers_upd = $task->getUpdater('suppliers');

		$this->loadTaxRulesGroupByRate();
	}
	
	protected function reloadCfg() {
	    parent::reloadCfg();
	    $cfg = $this->task->cfg;
		
	    $this->noover_categories = $cfg->get('NOOVER_PRODUCTS_CATEGORIES', false);
	    $this->noover_manufacturer = $cfg->get('NOOVER_PRODUCTS_MANUFACTURER', false);
		$this->noover_suppliers = $cfg->get('NOOVER_PRODUCTS_SUPPLIERS', false);
	    
	    $this->noover_name = $this->noover_content || $cfg->get('NOOVER_PRODUCTS_NAME', false);
	    $this->noover_descrip = $this->noover_content || $cfg->get('NOOVER_PRODUCTS_DESCRIP', false);
	    $this->noover_longdescrip = $this->noover_content || $cfg->get('NOOVER_PRODUCTS_LONGDESCRIP', false);
	    $this->noover_products_tags = $cfg->get('NOOVER_PRODUCTS_TAGS', false);
		$this->post_author = $cfg->get('POST_AUTHOR', false);
		$this->noover_visibility = $cfg->get('NOOVER_PRODUCTS_VISIBILITY', false);
	    
	}
	
	protected function dto2row($dto, $idx, $exists, $oldRowId) {
	    $ref = $this->matcher->dtoIdToStrFromDto($dto);
	    
		$rate = (string)($dto['iva'] * 100);
		if (!isset($this->tax_class_by_rate[$rate]))
		{
		    $msg = 'No se definió en la tienda un IVA del '.$rate.'% (prod. "'.$ref.'")';
			$task = $this->task;
			if ($task->stop_on_error) {
				throw new Fs2psException($msg);
			} else	{
				$this->task->log('ERROR: '.$msg);
				return null;
			}
		}
		
		$row = parent::dto2row($dto, $idx, $exists, $oldRowId);

		$replaceObjOrArray = array(
			'post_type' => 'product',
			'__ref' => $ref,
			'_tax_class' => $this->tax_class_by_rate[$rate],
			// '_visibility' => $this->enable ? 'visible' : 'hidden', // TODO Revisar
		);
		
		if ($this->post_author) {
			$replaceObjOrArray['post_author'] = $this->post_author;
		}
		
		Fs2psTools::replaceObjOrArray($row, $replaceObjOrArray);
		
		if ($this->enable===True) {
		    $row['post_status'] = 'publish';
		} else if (!$exists) {
		    $row['post_status'] = 'draft';
		} else { // if ($this->enable=='draft') {
			$old_status = get_post_status($oldRowId);
		    $row['post_status'] = $old_status=='trash'? $this->enable : $old_status;
		}
		
		if (isset($dto['name']) && (!$exists || !$this->noover_name)) $row['post_title'] = $dto['name'];
	    if (isset($dto['descrip']) && (!$exists || !$this->noover_descrip)) $row['post_excerpt'] = $dto['descrip'];
	    if (isset($dto['longdescrip']) && (!$exists || !$this->noover_longdescrip)) $row['post_content'] = $dto['longdescrip'];
	    if (isset($dto['slug']) && (!$exists || !$this->noover_url)) $row['post_name'] = $dto['slug'];
		if (isset($dto['virtual'])) $row['_virtual'] = $dto['virtual'];
		
		if (isset($dto['categories']))
		{
		    if (!$exists || !$this->noover_categories)
		        $row['__categories'] = $this->categories_upd->refsToIds($dto['categories']);
		}
		
		if (isset($dto['manufacturer']) && $this->manufacturers_upd->matcher->taxonomy)
		{
			if (!$exists || !$this->noover_manufacturer)
				$row['__manufacturers'] = $this->manufacturers_upd->refsToIds(array($dto['manufacturer']));
		}

		if (isset($dto['suppliers']) && $this->suppliers_upd->matcher->taxonomy)
		{
			if (!$exists || !$this->noover_suppliers)
				$row['__suppliers'] = $this->suppliers_upd->refsToIds($dto['suppliers']);
		}
		
		if (isset($dto['tags']) && (!$exists || !$this->noover_products_tags)) $row['__tags'] = $dto['tags'];
		
		if (isset($dto['combinations'])) $row['__combinations'] = $dto['combinations'];

		if (isset($dto['favourite'])) $row['__favourite'] = $dto['favourite'];

		if (!empty($dto['product_type'])) $row['__product_type'] = $dto['product_type'];

		return $row;
	}

	protected function insertOrUpdate($row, $exists, $oldRowId) {
		$id = parent::insertOrUpdate($row, $exists, $oldRowId);
		
		if (isset($row['__categories'])) {
		    $terms = $exists? get_the_terms($id, 'product_cat') : null;
		    $old_categories = $terms? array_map(function($o){return $o->term_id;}, $terms) : null;
		    //$old_categories = $terms? array_map(create_function('$o', 'return $o->term_id;'), $terms) : null;
		    $categories = $this->categories_upd->keepNotManagedIds($row['__categories'], $old_categories);
		    Fs2PsTools::wpErr(wp_set_object_terms($id, $categories, 'product_cat'));
		}
		if (empty($categories) && !$this->noover_categories && !$exists)
		{
		    $msg = 'WARN: El producto "'.$row['__ref'].'" no se asoció con ninguna categoría';
		    $this->task->log($msg);
		}
		
		foreach ($this->reltaxos as $k) {
			if (isset($row['__'.$k]) && $this->{$k.'_upd'}->matcher->taxonomy) {
				$taxonomy = $this->{$k.'_upd'}->matcher->taxonomy;
				$terms = $exists? get_the_terms($id, $taxonomy) : null;
				$old_rows = $terms? array_map(function($o){return $o->term_id;}, $terms) : null;
				$new_rows = $this->manufacturers_upd->keepNotManagedIds($row['__'.$k], $old_rows);
				Fs2PsTools::wpErr(wp_set_object_terms($id, $new_rows, $taxonomy));

				// Se se trata de una taxonomía de atributo, la añadimos a los atributos del producto
				if (strpos($taxonomy, 'pa_')===0) {
					$data = get_post_meta($id, '_product_attributes', true);
					if (empty($data)||!is_array($data)) $data = array();
					if ($new_rows) {
						$data[$taxonomy] = array(
							'name' => $taxonomy,
							'value' => '',
							'position' => 0,
							'is_visible' => 1,
							'is_taxonomy' => 1
						);		
					} else {
						unset($data[$taxonomy]);
					}
					update_post_meta($id, '_product_attributes', $data);
				}
			}
		}

		if (isset($row['__tags'])) Fs2PsTools::wpErr(wp_set_object_terms($id, $row['__tags'], 'product_tag'));
	
		// En principio se trata de un producto simple
		if ($this->hasToOverrideProductType($id)) {
			Fs2PsTools::wpErr(wp_set_object_terms($id, empty($row['__product_type'])? 'simple' : $row['__product_type'], 'product_type'));
		}
		
		if ($this->enable && !$this->noover_visibility) {
			# Hacemos el articulo siempre visible, independientemente del disabler que sea (hide/disable), a menos que se indique noover_visibility
			Fs2PsTools::wpErr(wp_remove_object_terms($id, array('exclude-from-catalog','exclude-from-search'), 'product_visibility'));
		}

		if (isset($row['__favourite'])) {
			if ($row['__favourite'] == true) {
				Fs2PsTools::wpErr(wp_set_object_terms($id, 'featured', 'product_visibility', true));
			} elseif ($row['__favourite'] == false) {
				Fs2PsTools::wpErr(wp_remove_object_terms($id, 'featured', 'product_visibility', true));
			}
		}
		
		return $id;
	}
	
	protected function onInsertedOrUpdated($dto, $row, $inserted)
	{
	    parent::onInsertedOrUpdated($dto, $row, $inserted);
		$id = $this->matcher->rowId($row);
	    if (isset($row['__combinations'])) {
	        $combinations_upd = $this->task->getUpdater('combinations');
	        if (empty($row['__combinations'])) {
	            $combinations_upd->removeNotPresentCombis($id, $row['__combinations'], true);
	        } else {
	            $combinations_upd->process($row['__combinations']);
	        }
	    }
		if (empty($row['__combinations'])) {
			Fs2psTools::wooClearCache($id, FALSE);
		} // else Fs2psTools::wooClearCache($id, FALSE); se invoca internamente en $combinations_upd->process

		Fs2psTools::wooQueueProductAttributeLookupUpdate($id);
	}
}

class Fs2psSizeColourCombinationUpdater extends Fs2psSKUUpdater {

	protected $products_upd;

	protected $attr_fields = array('size', 'colour');
	protected $field2type = array();
	protected $field2matcher = array();
	protected $attr_types = array();
	protected $noover_descrip;
	
	public function __construct($task, $name) {
	    parent::__construct($task, $name);
		
	    $this->products_upd = $task->getUpdater('products');
	    
	    for($i=0;$i<sizeof($this->attr_fields);$i++) {
	        $attr_field = $this->attr_fields[$i];
	        $updater = $task->getUpdater($attr_field.'s');
	        if ($updater) {
    	        $matcher = $updater->matcher;
    	        $attr_type = substr($matcher->taxonomy, 3);
    	        $this->attr_types[$i] = $attr_type;
    	        $this->field2type[$attr_field] = $attr_type;
    	        $this->field2matcher[$attr_field] = $matcher;
	        }
	    }
	}

	protected function reloadCfg() {
	    parent::reloadCfg();
	    $cfg = $this->task->cfg;
		$this->noover_descrip = $this->noover_content || $cfg->get('NOOVER_PRODUCTS_DESCRIP', false);
	}

	protected function dto2row($dto, $idx, $exists, $oldRowId) {
		$row = parent::dto2row($dto, $idx, $exists, $oldRowId);
		
		$parent_id = $this->products_upd->matcher->rowIdFromDtoId($dto['ref']);
		$post_title = Fs2psTools::dbValue('select post_title from @DB_posts where ID='.$parent_id);
		if (!empty($dto['colour'])) $post_title .= ' - '.$dto['colour'];
		if (!empty($dto['size'])) $post_title .= ' - '.$dto['size']; 

		if (isset($dto['descrip']) && (!$exists || !$this->noover_descrip)) $row['_variation_description'] = $dto['descrip'];
		
		Fs2psTools::replaceObjOrArray($row, array(
			'post_title' => $post_title,
			'post_parent'   => $parent_id,
			'post_type'     => 'product_variation',
			'_tax_class' => 'parent',
			//'guid'          =>  home_url() . '/?product_variation='.$name, //product-91-variation
			// '_visibility' => 'visible',
				
			// Extra info to be used in insertOrUpdate
		    '__size' => isset($dto['size']) ? $dto['size'] : null, // sanitize_title($dto['size']) : null,
		    '__colour' => isset($dto['colour']) ? $dto['colour'] : null, // sanitize_title($dto['colour'])
		    
		    // Siempre visible. La habilitación deshabilitación se gestionará en el producto padre
		    'post_status' => 'publish',
		    '_visibility' => 'visible',
			'_virtual' => isset($dto['virtual']) ?  $dto['virtual'] : 'no',
		));
		
		return $row;
	}

	protected function insertOrUpdate($row, $exists, $oldRowId) {
		$id = parent::insertOrUpdate($row, $exists, $oldRowId);
		
		foreach ($this->attr_fields as $attr_field)  {
			$attr_type = $this->field2type[$attr_field];
			if (empty($row['__'.$attr_field])) {
				delete_post_meta($id,'attribute_pa_'.$attr_type);
			} else {
				$attr_type_matcher = $this->field2matcher[$attr_field];
				$term = Fs2PsTools::wpErr(get_term( $attr_type_matcher->rowIdFromDtoId($row['__'.$attr_field]), 'pa_'.$attr_type));
				update_post_meta($id, 'attribute_pa_'.$attr_type, $term->slug);
			}
		}
		
		Fs2psTools::wooClearCache($id, TRUE);
		
		return $id;
	}
	
	public function removeNotPresentCombis($parent_id, $combinations, $simulate_full_process=false)
	{
	    $all_combis = Fs2psTools::dbSelect('
			select p.ID from `@DB_posts` p
			where
				p.post_parent = '.(int)$parent_id.' and
				p.post_type=\'product_variation\'
		');
	    if (!empty($all_combis)) {
	        if ($simulate_full_process) {
	            $this->resetCounters($combinations);
	        }
	        
	        $matcher = $this->matcher;
	        $updated_combi_ids = array();
	        foreach ($combinations as $combi) {
	            $updated_combi_ids[] = $matcher->rowId($combi);
	        }
	        foreach ($all_combis as $combi) {
	            $combi_id = $matcher->rowId($combi);
	            if (!in_array($combi_id, $updated_combi_ids)) {
	                Fs2PsTools::wpErr(wp_delete_post($combi_id, true));
	                $this->ndeleted++;
	            }
	        }
	        
	        if ($simulate_full_process) {
	            $this->logProcess();
	        }
	    }
	}

	protected function sameGroup($row_a, $row_b) {
		return $row_a['post_parent'] == $row_b['post_parent'];
	}

	protected function onGroupUpdated($group_rows) {
		if (!empty($group_rows))
		{
		    $row = $group_rows[0];
			$parent_id = $row['post_parent'];
			
			// Si tiene combinaciones se trata de un producto variable
			if ($this->hasToOverrideProductType($parent_id)) {
				Fs2PsTools::wpErr(wp_set_object_terms($parent_id, 'variable', 'product_type'));
			}
			
			// Gestionaremos el stock en las variaciones, no en el producto padre
			update_post_meta($parent_id, '_manage_stock', 'no'); // TODO Revisar cuando manage_stock_ifexist && $exist
			if ($row['_manage_stock']=='yes') {
    			if (isset($row['_stock'])) {
        			// TODO: && habilitada_gestion_existencias
        			// Habrá existencias en el padre si hay alguna combinación con stock.
        			// Si no hay ninguna combinacion con stock nos preguntamos si se permite reservar.
        			$any_combi_instock = in_array(TRUE, array_map(function($v) { return $v['_stock']>0; }, $group_rows));
        			$any_combi_backorders = $any_combi_instock? FALSE : in_array(TRUE, array_map(function($v) { return !empty($v['__backorders']) && $v['__backorders']!='no'; }, $group_rows));
        			$parent_stock_status = $any_combi_instock? 'instock' : ($any_combi_backorders? 'onbackorder' : 'outofstock');
    			} else {
    			    // Get the stock quantity sum of all product variations (children)
    			    $stock_quantity = Fs2psTools::dbValue("
                        SELECT SUM(pm.meta_value)
                        FROM @DB_posts as p
                        JOIN @DB_postmeta as pm ON p.ID = pm.post_id
                        WHERE p.post_type = 'product_variation'
                        AND p.post_status = 'publish'
                        AND p.post_parent = '$parent_id'
                        AND pm.meta_key = '_stock'
                        AND pm.meta_value IS NOT NULL AND pm.meta_value>0
                    ");
    			    
    			    $parent_stock_status = $stock_quantity? 'instock' : 'outofstock';
    			}
    			update_post_meta($parent_id, '_stock_status', $parent_stock_status);
    			if ($parent_stock_status=='outofstock') Fs2PsTools::wpErr(wp_set_post_terms($parent_id, 'outofstock', 'product_visibility', true));
    			else Fs2PsTools::wpErr(wp_remove_object_terms($parent_id, 'outofstock', 'product_visibility'));
			} else {
			    update_post_meta($parent_id, '_stock_status', 'instock');
			    Fs2PsTools::wpErr(wp_remove_object_terms($parent_id, 'outofstock', 'product_visibility'));
			}
			
			// XXX: Revisar
			// Fill selectable dropdowns
			$attr_values_by_type = array();
			foreach ($this->attr_fields as $attr_field) {
				$attr_type = $this->field2type[$attr_field];
				$taxonomy = 'pa_'.$attr_type;
				$attr_type_matcher = $this->field2matcher[$attr_field];
				foreach ($group_rows as $row) {
					if (!empty($row['__'.$attr_field])) {
						if (!isset($attr_values_by_type[$attr_type])) {
							$attr_values_by_type[$attr_type] = array();
						}
						if (!in_array($row['__'.$attr_field], $attr_values_by_type[$attr_type])) {
						    $attr_values_by_type[$attr_type][] = get_term( $attr_type_matcher->rowIdFromDtoId($row['__'.$attr_field]), $taxonomy)->slug;
						}
					}
				}
				
				if (!empty($attr_values_by_type[$attr_type])) {
					Fs2PsTools::wpErr(wp_set_object_terms(
					    $parent_id, $attr_values_by_type[$attr_type], $taxonomy, true));
				}
			}
			
			// Save fs2ps attributes
			$product_attributes = get_post_meta($parent_id, '_product_attributes', true);
			if ($product_attributes) { // Hacemos hueco al principio
				$posinc = sizeof($attr_values_by_type);
				foreach ($this->attr_types as $attr_type) {
					unset($product_attributes['pa_'.$attr_type]);
				}
				foreach ($product_attributes as $k => $v) {
					$pos = empty($v['position'])? 0 : $v['position'];  
					$v['position'] = $pos + $posinc;
				}
			} else $product_attributes = array(); 
			$position = 0;
			foreach ($attr_values_by_type as $attr_type => $attr_value) {
				$product_attributes['pa_'.$attr_type] = array(
					'name' => 'pa_'.$attr_type,
					'value' => '', // TODO falta valor escogido?: $attr_value
					'position' => (string)$position,
					'is_visible' => '1',
					'is_variation' => '1',
					'is_taxonomy' => '1'
				);
				$position++;
			}
			update_post_meta($parent_id, '_product_attributes', $product_attributes);
			
			// TODO cfillol: Set default combination if none
			
			// Remove not uploaded combinations
			$this->removeNotPresentCombis($parent_id, $group_rows);

			Fs2psTools::wooClearCache($parent_id, FALSE);
			if ($this->rbp_activated) wc_rbp_update_variations_data($parent_id);
		}

	}

}

class Fs2psUpdateProductUpdater extends Fs2psProductUpdater
{
    
    protected function dto2row($dto, $idx, $exists, $oldRowId)
    {
        if (!$exists) return null;
        
        $row = array();
        if ($this->manage_stock && (!$exists || !$this->manage_stock_ifnotexist)) {
			if (isset($dto['stock'])) $row['_stock'] = $dto['stock'];
        	// Si no se gestiona stock no es necesario actualizar combinaciones que sólo actualizan stock
        	if (isset($dto['combinations'])) $row['__combinations'] = $dto['combinations'];
		}
        return $row;
    }
    
    protected function insertOrUpdate($row, $exists, $oldRowId)
    {
        // if (!$exists) return null; // Ya se controla en dto2row
        
        if (isset($row['_stock'])) {
            Fs2psTools::dbUpdate('postmeta', array('meta_value'=>$row['_stock']), array('post_id'=>$oldRowId, 'meta_key'=>'_stock'));
        }
        
        if ($this->enable) {
			if ($this->enable==='draft')
				Fs2psTools::dbUpdate('posts', array('post_status'=>'draft'), array('ID'=>$oldRowId, 'post_status'=>'trash'));
			else
            	Fs2psTools::dbUpdate('posts', array('post_status'=>'publish'), array('ID'=>$oldRowId));
            Fs2psTools::dbUpdate('postmeta', array('meta_value'=>'visible'), array('post_id'=>$oldRowId, 'meta_key'=>'_visibility'));
        }

        return $oldRowId;
    }
    
    protected function onInsertedOrUpdated($dto, $row, $inserted)
    {
		$id = $this->matcher->rowId($row);
        $this->matcher->updateMatch($this->matcher->dtoId($dto), $id);
        if (!empty($row['__combinations'])) {
            $combinations_upd = $this->task->getUpdater('combinations');
            $combinations_upd->process($row['__combinations']);
        }

		Fs2psTools::wooClearCache($id, FALSE);
    }
    
    protected function onProcessStart($dtos) { }
    protected function onProcessEnd($dtos) { }
    
}


class Fs2psUpdateSizeColourCombinationUpdater extends Fs2psSizeColourCombinationUpdater
{
    protected function dto2row($dto, $idx, $exists, $oldRowId)
    {
        if (!$exists) return null;
        
        $row = array();
        if ($this->manage_stock && (!$exists || !$this->manage_stock_ifnotexist) && isset($dto['stock'])) $row['_stock'] = $dto['stock'];
        
        $parent_id = $this->products_upd->matcher->rowIdFromDtoId($dto['ref']);
        if (empty($parent_id)) {
            throw new Fs2psException(
                'No existe el producto referenciado por la combinación: '.$this->getSizeColDesc($dto)
            );
        }
        $row['post_parent'] = $parent_id;
        
        return $row;
    }
    
    protected function insertOrUpdate($row, $exists, $oldRowId)
    {
        if (isset($row['_stock'])) {
            Fs2psTools::dbUpdate('postmeta', array('meta_value'=>$row['_stock']), array('post_id'=>$oldRowId, 'meta_key'=>'_stock'));
            Fs2psTools::wooClearCache($oldRowId, TRUE);
        }
        return $oldRowId;
    }
    
    // Sólo actulizamos, así que no crearemos nada ni borraremos nada
    protected function onGroupUpdated($group_rows) { }
    protected function onProcessStart($dtos) { }
    protected function onProcessEnd($dtos) { }
}


class Fs2psStockablesUpdater extends Fs2psSKUUpdater {
    
    protected $productMatcher;
    protected $combiMatcher;
    protected $parentsData = array();
	protected $ean_metakey;
	protected $custom_fields_metakeys;
	protected $update_ean;
	protected $update_custom_fields;
    
    public function __construct($task, $name)
    {
        parent::__construct($task, $name);
        $this->productMatcher = Fs2psMatcherFactory::get($task, 'products');
        $this->combiMatcher = Fs2psMatcherFactory::get($task, 'combinations');
        $this->loadTaxRulesGroupByRate();
    }

	protected function reloadCfg() 
    {
        parent::reloadCfg();
        $cfg = $this->task->cfg;
        $this->update_ean = $cfg->get('UPDATE_STOCKABLES_EAN', FALSE);
		$this->ean_metakey = $cfg->get('EAN_METAKEY', '_wpm_gtin_code');
		$this->update_stockables_tax = $cfg->get('UPDATE_STOCKABLES_TAX', FALSE);

		$this->custom_fields_metakeys = $cfg->get('CUSTOM_FIELDS_METAKEYS');
		$this->update_custom_fields = $cfg->get('UPDATE_STOCKABLES_CUSTOM_FIELDS');
    }
    
    protected function dto2row($dto, $idx, $exists, $oldRowId, $es_combi=FALSE) {
        $matcher = $es_combi? $this->combiMatcher : $this->productMatcher;
        $ref = $matcher->dtoIdToStrFromDto($dto);
        
        if (isset($dto['prices']) || isset($dto['price'])) {
            $rate = (string)($dto['iva'] * 100);
            if (!isset($this->tax_class_by_rate[$rate]))
            {
                $msg = 'No se definió en la tienda un IVA del '.$rate.'% (prod. "'.$ref.'")';
                $task = $this->task;
                if ($task->stop_on_error) {
                    throw new Fs2psException($msg);
                } else	{
                    $this->task->log('ERROR: '.$msg);
                    return null;
                }
            }
        
        }
        
        $row = parent::dto2row($dto, $idx, $exists, $oldRowId);
        
        // Evitamos modificaciones indeseables en modo update_stockables
        unset($row['post_status']);
        unset($row['_visibility']);
        unset($row['_sku']);
        //$row['_sku'] = $matcher->referenceFromDto($dto);
        
        if (isset($dto['prices']) || isset($dto['price'])) {
			$rplcArr = array(
                '__ref' => $ref,
            );
			// Actualizamos el tipo de impuesto en función del impuesto que nos llega de Fs
			if ($this->update_stockables_tax) {
				$rplcArr['_tax_class'] = $es_combi? 'parent' : $this->tax_class_by_rate[$rate]; # parent si es combi
			}
            Fs2psTools::replaceObjOrArray($row, $rplcArr);

        } else {
            Fs2psTools::replaceObjOrArray($row, array('__ref' => $ref, ));
        }
        
        return $row;
    }
    
    protected function processOne($dto, $post_id, $dto_id, $es_combi)
    {
		if (!(Fs2psTools::dbValue('
			select count(1) from `@DB_posts` 
			where ID = '.(int)$post_id.' and (post_type=\'product\' or post_type=\'product_variation\')')
		)) {
			$this->task->log('WARN: No existe '.($es_combi? 'la combinación': 'el producto').' '.$dto_id);
			return FALSE;
		}
		
        $parent_id = null;
        if ($es_combi) {
            // En versiones anteriores se podían producir falsos positivos de
            // combinaciones porque tanto productos como combinaciones usan la tabla posts
            // y entonces no se discriminaba por post_type='product_variation' en matcher de combinaciones.
            $parent_id = wp_get_post_parent_id($post_id);
            if(!$parent_id) $es_combi = FALSE;
        }
        
        $row = $this->dto2row($dto, 0, TRUE, $post_id, $es_combi);
        $matcher = $es_combi? $this->combiMatcher : $this->productMatcher;
        $row[$matcher->row_id_field] = $post_id;

		$ean_metakey = $this->ean_metakey;
		if($this->update_ean && !empty($ean_metakey) && isset($dto['ean']) && !($this->update_ean ==='ifnotempty' && empty($dto['ean']))) {
			update_post_meta($post_id, $ean_metakey, $dto['ean']);
		}

		$custom_fields_metakeys = $this->custom_fields_metakeys;
		if ($this->update_custom_fields && !empty($custom_fields_metakeys) && isset($dto['custom_fields'])) {
			foreach ($custom_fields_metakeys as $custom_fields_metakey) {
				update_post_meta($post_id, $custom_fields_metakey, $dto['custom_fields']);
			}
		}
        
        $this->insertOrUpdate($row, TRUE, $post_id);
        
        Fs2psTools::wooClearCache($post_id, $es_combi);
        
        if ($es_combi) { // && $parent_id
            $parentsData = &$this->parentsData;
            if (!isset($parentsData[$parent_id])) $parentsData[$parent_id] = array();
            $pdata = &$parentsData[$parent_id];
            if ($row['_manage_stock'] == 'no') $pdata['stock'] = 1;
            else if (isset($row['_stock'])) {
                $pdata['stock'] = !empty($pdata['stock']) && $pdata['stock']>0 || $row['_stock']>0? 1 : 0;
				if (!$pdata['stock']) {
					// Si no hay stock en esta combinación, comprobamos si hay stock en alguna otra combinación.
					// Si hay stock en alguna combinación, se considera que hay stock en el producto padre.
					$pdata['stock'] = Fs2psTools::dbValue('
						SELECT sum(cm.meta_value)
						FROM @DB_posts c
						inner join @DB_postmeta cm on cm.meta_key=\'_stock\' and cm.post_id=c.ID
						where post_parent='.$parent_id.' and cm.meta_value>0
					')>0? 1 : 0;
				}	
                if ($row['_stock']<=0 && !empty($row['__backorders']) && $row['__backorders']!='no') $pdata['backorders'] = 1;
            }
            if (isset($row['__prices'])) $pdata['prices'] = 1;
        }
		return TRUE;
    }
    
    public function process($dtos, $force_combi=false)
    {
        $this->ntotal += $dtos==null? 0 : count($dtos);
        $nupdated_products = 0;
        $nupdated_products_repe = 0;
        $nupdated_combis = 0;
        $nupdated_combis_repe = 0;
        $parentsData = &$this->parentsData;
        
        if ($dtos) {
            $combiMatcher = $this->combiMatcher;
            $productMatcher = $this->productMatcher;
            foreach ($dtos as $dto)
            {
                if (!empty($dto['combinations'])) {
                    $this->process($dto['combinations'], !$this->update_stockables_combis_can_be_products);
                    continue;
                }
                
				// Hace match con combinaciones?				
				$combiDtoId = $combiMatcher->dtoId($dto);
				$combiDtoIdStr = $combiDtoId? $combiMatcher->dtoIdToStr($combiDtoId) : '';

				$combiPostIds = null;
				if (!empty($combiDtoIdStr)) {
					try {
						$combiPostIds = $combiMatcher->rowIdsFromDto($dto);
					} catch (Exception $e) { }
				}

				if (empty($combiPostIds)) {
					if ($force_combi) {
						// No existe combinación para esa ref y debería existir
						$this->task->log('WARN: No existe la combinación ('.$combiDtoIdStr.')');
						continue;
					}
				} else {
					$any_repe_updated = FALSE;
					foreach ($combiPostIds as $combiPostId) {
						if ($this->processOne($dto, $combiPostId, $combiDtoIdStr, TRUE)) {
							$any_repe_updated = TRUE;
							$nupdated_combis_repe++;
						}
					}
					if ($any_repe_updated) $nupdated_combis++;
				}
				
				// Hace match con productos?
				if (!$force_combi) { // Evitamos falsos matchs por ref cuando son dtos de combis. Puede que sea prod en FS y combi en PS pero consideramos que nunca pasará que es combi en FS y product en PS.
					$productDtoId = $productMatcher->dtoId($dto);
					$productDtoIdStr = $productDtoId? $productMatcher->dtoIdToStr($productDtoId) : '';
					
					$productPostIds = null;
					if (!empty($productDtoIdStr)) {
						try {
							$productPostIds = $productMatcher->rowIdsFromDto($dto);
						} catch (Exception $e) { }
					}
					
					if (empty($productPostIds)) {
						if (empty($combiPostIds)) {
							// No existe producto ni combinación para esa ref
							$combiDtoIdStr = $combiDtoId? $combiMatcher->dtoIdToStr($combiDtoId) : '';
							$this->task->log('WARN: No existe producto ('.$productDtoIdStr.') ni combinación ('.$combiDtoIdStr.')');
							continue;
						}
					} else {
						$any_repe_products = FALSE;
						foreach ($productPostIds as $productPostId) {
							if ($this->processOne($dto, $productPostId, $productDtoIdStr, FALSE)) {
								$any_repe_products = TRUE;
								$nupdated_products_repe++;
							}
						}
						if ($any_repe_products) $nupdated_products++;
					}
				}
                
                $this->nprocessed++;
            }
        }
        
        if (!$force_combi) {
            foreach ($parentsData as $parent_id=>$pdata) {
                if (isset($pdata['stock'])) {
                    update_post_meta($parent_id, '_manage_stock', 'no');
                    $stock_status = $pdata['stock']>0? 'instock' : (empty($pdata['backorders'])? 'outofstock' : 'onbackorder');
                    update_post_meta($parent_id, '_stock_status', $stock_status);
                    if ($stock_status=='outofstock') Fs2PsTools::wpErr(wp_set_post_terms($parent_id, 'outofstock', 'product_visibility', true));
                    else Fs2PsTools::wpErr(wp_remove_object_terms($parent_id, 'outofstock', 'product_visibility'));
                    Fs2psTools::wooClearCache($parent_id, FALSE);
                }
                if (isset($pdata['prices'])) {
                    if (!isset($pdata['stock'])) Fs2psTools::wooClearCache($parent_id, FALSE);
                    if ($this->rbp_activated) wc_rbp_update_variations_data($parent_id);
                }
            }
        }
        
        if ($nupdated_combis) $this->task->log('combinations: '.$nupdated_combis.' actualizados'.($nupdated_combis_repe>$nupdated_combis? ' (+'.($nupdated_combis_repe-$nupdated_combis).' reps)' : ''));
        if ($nupdated_products) $this->task->log('products: '.$nupdated_products.' actualizados'.($nupdated_products_repe>$nupdated_products? ' (+'.($nupdated_products_repe-$nupdated_products).' reps)' : ''));
    }
    
}

class Fs2psStockablesCombisUpdater extends Fs2psStockablesUpdater {
    public function process($dtos, $force_combi=false)
    {
        parent::process($dtos, true);
    }
}

// XXX: En pedidos sólo soportamos actualizaciones, al menos de momento. No existe Fs2psOrderUpdater.
class Fs2psUpdateOrderUpdater extends Fs2psUpdater
{
	protected $updatable_status;

	protected function getFailSafeStatuses($statuses, $filter='/[a-z\-]+/') {
		if (empty($statuses)) return null;
		$matches = null;
		preg_match_all($filter, $statuses, $matches);
		return preg_replace('/^wc\-/i', '', $matches[0]);
	}

    protected function reloadCfg() {
		parent::reloadCfg();
		$cfg = $this->task->cfg;
		$updatable_status = strtolower($cfg->get(strtoupper($this->name).'_UPDATABLE_STATUS', ''));
        $this->updatable_status = $this->getFailSafeStatuses($updatable_status, '/[a-z0-9\-]+/i');
	}

    protected function dto2row($dto, $idx, $exists, $oldRowId)
    {
        if (!$exists) return null;
        if (empty($dto['status'])) return null;
        
		$statuses = $this->getFailSafeStatuses($dto['status']);
		if (!$statuses || !$statuses[0]) return null;
		
		return array('_status' => $statuses[0]);
    }
    
    protected function insertOrUpdate($row, $exists, $oldRowId, $oldObj=null)
    {
		$order = new WC_Order($oldRowId);
		if (empty($order)) {
			$this->task->log('INFO: El pedido '.$oldRowId.' no existe');
			return;
		}

		$actual_status = $order->get_status();

		if ($this->updatable_status && !in_array($actual_status, $this->updatable_status)) {
			$this->task->log('INFO: El pedido '.$oldRowId.' no tiene un estado actualizable');
			return;
		}
		
		$new_status = $row['_status'];
		if ($actual_status==$new_status) {
			$this->task->log('INFO: El pedido '.$oldRowId.' ya está en estado '.$new_status);
		} else {
			$order->update_status($new_status);
		}
    }
}
